#!/usr/bin/env python3
import re
import tkinter as tk
from tkinter import ttk, filedialog, messagebox, simpledialog
from decimal import Decimal, ROUND_HALF_UP
from pathlib import Path
import shutil

UPDATE_RE = re.compile(
    r'^(update_field)\s+tires\s+(\S+)\s+GRIP_SCALE\s+(Front|Rear)\s+([0-9]+(?:\.[0-9]+)?)\s*$',
    re.IGNORECASE
)

class ParsedEntry:
    def __init__(self, category, car, side, value, line_index, raw_line):
        self.category = category
        self.car = car
        self.side = side
        self.value = float(value)
        self.line_index = line_index
        self.raw_line = raw_line

    def to_line(self):
        d = Decimal(str(self.value)).quantize(Decimal('0.000001'), rounding=ROUND_HALF_UP)
        s = format(d.normalize(), 'f')
        return f"update_field tires {self.car} GRIP_SCALE {self.side} {s}"

class GripFile:
    def __init__(self):
        self.path = None
        self.file_lines = []
        self.entries = []
        self.categories = []
        self.line_to_entry_indexes = {}

    def load(self, path):
        self.path = Path(path)
        self.file_lines = self.path.read_text(encoding='utf-8').splitlines(keepends=True)
        self.entries = []
        self.line_to_entry_indexes = {}
        self.categories = []

        current_category = "Uncategorized"
        self.categories.append(current_category)

        for i, raw in enumerate(self.file_lines):
            stripped = raw.strip()
            if stripped == "":
                continue

            m = UPDATE_RE.match(stripped)
            if m:
                car = m.group(2)
                side = m.group(3)
                val = m.group(4)
                entry = ParsedEntry(current_category, car, side, val, i, raw)
                idx = len(self.entries)
                self.entries.append(entry)
                self.line_to_entry_indexes.setdefault(i, []).append(idx)
            else:
                current_category = stripped
                if not self.categories or self.categories[-1] != current_category:
                    self.categories.append(current_category)

    def apply_bulk(self, category_filter, target_side, delta_value, mode_percent=False):
        for e in self.entries:
            if category_filter != "All" and e.category != category_filter:
                continue
            if target_side != "Both" and e.side.lower() != target_side.lower():
                continue
            old = e.value
            if mode_percent:
                e.value = old * (1.0 + (delta_value / 100.0))
            else:
                e.value = old + delta_value

    def update_single(self, entry_index, new_value):
        self.entries[entry_index].value = float(new_value)

    def save(self, out_path=None):
        if out_path is None:
            out_path = self.path
        out_path = Path(out_path)
        if out_path == self.path:
            bak = self.path.with_suffix(self.path.suffix + '.bak')
            shutil.copy2(self.path, bak)

        new_lines = list(self.file_lines)

        for e in self.entries:
            formatted = e.to_line()
            orig = self.file_lines[e.line_index]
            if orig.endswith('\r\n'):
                newline = '\r\n'
            elif orig.endswith('\n'):
                newline = '\n'
            else:
                newline = ''
            new_lines[e.line_index] = formatted + newline

        out_path.write_text(''.join(new_lines), encoding='utf-8')

class GripEditorApp(tk.Tk):
    def __init__(self):
        super().__init__()
        self.title("Tireed")
        self.geometry("1000x650")
        self.gripfile = GripFile()
        self.create_widgets()

    def create_widgets(self):
        topfrm = ttk.Frame(self)
        topfrm.pack(fill='x', padx=8, pady=6)

        ttk.Button(topfrm, text="Open File...", command=self.open_file).pack(side='left')
        self.lbl_path = ttk.Label(topfrm, text="No file loaded", foreground='blue')
        self.lbl_path.pack(side='left', padx=10)

        ttk.Button(topfrm, text="Save", command=self.save_file).pack(side='right')
        ttk.Button(topfrm, text="Reload", command=self.reload_file).pack(side='right', padx=6)

        midfrm = ttk.Frame(self)
        midfrm.pack(fill='both', expand=True, padx=8, pady=6)

        left = ttk.Frame(midfrm, width=300)
        left.pack(side='left', fill='y', padx=(0,10))

        ttk.Label(left, text="Category / Filter").pack(anchor='w')
        self.cat_var = tk.StringVar(value="All")
        self.cat_combo = ttk.Combobox(left, textvariable=self.cat_var, state='readonly')
        self.cat_combo.pack(fill='x')

        ttk.Separator(left).pack(fill='x', pady=6)
        ttk.Label(left, text="Bulk operation").pack(anchor='w', pady=(6,0))

        self.side_var = tk.StringVar(value="Front")
        ttk.Radiobutton(left, text="Front", variable=self.side_var, value="Front").pack(anchor='w')
        ttk.Radiobutton(left, text="Rear", variable=self.side_var, value="Rear").pack(anchor='w')
        ttk.Radiobutton(left, text="Both", variable=self.side_var, value="Both").pack(anchor='w')

        ttk.Label(left, text="Delta value").pack(anchor='w', pady=(8,0))
        self.delta_var = tk.StringVar(value="0.15")
        ttk.Entry(left, textvariable=self.delta_var).pack(fill='x')

        self.mode_var = tk.StringVar(value="absolute")
        ttk.Radiobutton(left, text="Absolute (add/subtract value)", variable=self.mode_var, value="absolute").pack(anchor='w', pady=(6,0))
        ttk.Radiobutton(left, text="Percent (e.g. 15 = +15%)", variable=self.mode_var, value="percent").pack(anchor='w')

        ttk.Button(left, text="Apply Bulk", command=self.apply_bulk).pack(fill='x', pady=(10,4))
        ttk.Button(left, text="Refresh Table", command=self.refresh_table).pack(fill='x')

        ttk.Separator(left).pack(fill='x', pady=8)
        ttk.Label(left, text="Selected line actions").pack(anchor='w')
        ttk.Button(left, text="Edit Selected", command=self.edit_selected).pack(fill='x', pady=(6,4))
        ttk.Button(left, text="Revert Selected to file", command=self.revert_selected).pack(fill='x')

        ttk.Separator(left).pack(fill='x', pady=8)
        ttk.Label(left, text="notes").pack(anchor='w')
        notes = (
            "• select category to edit car classes (carbon) in bulk\n"
            "• bulk edits only modify visible entries\n"
            "• double click a row to edit manually"
        )
        ttk.Label(left, text=notes, wraplength=260, justify='left').pack(anchor='w', pady=(4,0))

        right = ttk.Frame(midfrm)
        right.pack(fill='both', expand=True)

        cols = ('idx','category','car','side','value','line')
        self.tree = ttk.Treeview(right, columns=cols, show='headings', selectmode='browse')
        self.tree.heading('idx', text='#')
        self.tree.heading('category', text='Category')
        self.tree.heading('car', text='Car')
        self.tree.heading('side', text='Side')
        self.tree.heading('value', text='Value')
        self.tree.heading('line', text='File line #')
        self.tree.column('idx', width=40, anchor='center')
        self.tree.column('category', width=200)
        self.tree.column('car', width=150)
        self.tree.column('side', width=70, anchor='center')
        self.tree.column('value', width=80, anchor='center')
        self.tree.column('line', width=80, anchor='center')
        self.tree.pack(fill='both', expand=True)

        self.tree.bind("<Double-1>", lambda e: self.edit_selected())

    def open_file(self):
        path = filedialog.askopenfilename(title="Open grip update file", filetypes=[("nfsms","*.nfsms")])
        if not path:
            return
        self.gripfile.load(path)
        self.lbl_path.config(text=str(self.gripfile.path))
        items = ["All"] + [c for c in self.gripfile.categories if c.strip() != ""]
        self.cat_combo['values'] = items
        self.cat_combo.set("All")
        self.refresh_table()

    def reload_file(self):
        if not self.gripfile.path:
            return
        self.gripfile.load(self.gripfile.path)
        self.cat_combo['values'] = ["All"] + [c for c in self.gripfile.categories if c.strip() != ""]
        self.cat_combo.set("All")
        self.refresh_table()

    def refresh_table(self):
        for r in self.tree.get_children():
            self.tree.delete(r)
        for idx, e in enumerate(self.gripfile.entries):
            self.tree.insert('', 'end', iid=str(idx),
                values=(idx, e.category, e.car, e.side, f"{e.value:.6f}".rstrip('0').rstrip('.'), e.line_index+1)
            )

    def apply_bulk(self):
        if not self.gripfile.path:
            return
        cat = self.cat_var.get()
        side = self.side_var.get()
        try:
            delta = float(self.delta_var.get().strip())
        except:
            return
        mode_percent = (self.mode_var.get() == "percent")

        self.gripfile.apply_bulk(cat, side, delta, mode_percent)
        self.refresh_table()

    def edit_selected(self):
        sel = self.tree.selection()
        if not sel:
            return
        idx = int(sel[0])
        entry = self.gripfile.entries[idx]
        new = simpledialog.askstring("Edit value", f"{entry.value}", initialvalue=str(entry.value), parent=self)
        if new is None:
            return
        try:
            nv = float(new)
        except:
            return
        self.gripfile.update_single(idx, nv)
        self.refresh_table()

    def revert_selected(self):
        sel = self.tree.selection()
        if not sel:
            return
        idx = int(sel[0])
        entry = self.gripfile.entries[idx]
        m = UPDATE_RE.match(entry.raw_line.strip())
        if not m:
            return
        orig_val = float(m.group(4))
        self.gripfile.update_single(idx, orig_val)
        self.refresh_table()

    def save_file(self):
        if not self.gripfile.path:
            return
        out = self.gripfile.path
        self.gripfile.save(out)

if __name__ == "__main__":
    app = GripEditorApp()
    app.mainloop()
